Вот переписанная функция с использованием `match` (доступен в PHP 8.0+), которая возвращает стили для HTTP-методов:

### **1. Функция `getMethodColor` с `match`**
```php
function getMethodColor(string $method): string
{
    return match(strtoupper($method)) {
        'GET'    => 'background: #E6F7FF; color: #003366;',
        'POST'   => 'background: #E8F5E9; color: #1B5E20;',
        'PUT', 
        'PATCH'  => 'background: #FFF3E0; color: #E65100;', // Один стиль для PUT и PATCH
        'DELETE' => 'background: #FFEBEE; color: #C62828;',
        default  => 'background: #F5F5F5; color: #212121;'
    };
}
```

### **2. Полный пример таблицы с `match`**
```php
function getRoutingTable(array $routingRules): string
{
    $table = "<table border='1' cellpadding='5' cellspacing='0' style='width:100%; border-collapse: collapse;'>";
    $table .= "<thead><tr style='background: #f0f0f0;'>
        <th>Method + Route</th>
        <th>Handler</th>
        <th>Name</th>
        <th>Before Middlewares</th>
        <th>After Middlewares</th>
    </tr></thead><tbody>";

    foreach ($routingRules as $route) {
        $method = $route['httpMethod'];
        $style = getMethodColor($method);
        
        $handler = match(true) {
            is_array($route['handler']) && count($route['handler']) === 2 
                => "{$route['handler'][0]}@{$route['handler'][1]}",
            $route['handler'] instanceof Closure 
                => 'Closure',
            isset($route['handler']['__invoke']) 
                => 'Invokable Class',
            default => 'Unknown'
        };

        $table .= "<tr style='{$style}'>
            <td><strong>{$method}</strong> {$route['route']}</td>
            <td>{$handler}</td>
            <td>{$route['name'] ?? 'NULL'}</td>
            <td>" . formatMiddlewares($route['middlewares']['before']) . "</td>
            <td>" . formatMiddlewares($route['middlewares']['after']) . "</td>
        </tr>";
    }
    
    $table .= "</tbody></table>";
    return $table;
}

// Вспомогательная функция для форматирования middleware
function formatMiddlewares(iterable $middlewares): string
{
    $result = [];
    foreach ($middlewares as $mw) {
        if (is_array($mw) && count($mw) === 2) {
            $result[] = "{$mw[0]}@{$mw[1]}";
        }
    }
    return $result ? implode('<br>', $result) : 'None';
}
```

### **Ключевые улучшения**:
1. **`match` для HTTP-методов**:
   - Четкое сопоставление значений без `break`.
   - Группировка `PUT` и `PATCH` в один стиль.

2. **`match` для определения типа handler**:
   - Заменил цепочку `if-else` на более читаемый `match(true)`.

3. **Дополнительные стили**:
   - Улучшено оформление таблицы (границы, отступы).
   - Жирный шрифт для HTTP-метода.

4. **Вспомогательная функция**:
   - `formatMiddlewares()` для единообразного вывода middleware.

### **Пример вывода**:
| Method + Route               | Handler                          | Name       | Before Middlewares       | After Middlewares |
|------------------------------|----------------------------------|------------|--------------------------|-------------------|
| **GET** /api/users           | UserController@index             | users.list | AuthMiddleware           | None              |
| **POST** /api/users          | UserController@store             | NULL       | AuthMiddleware, CSRF     | LogMiddleware     |

Такой код лучше соответствует современным стандартам PHP (8.0+) и улучшает читаемость.