#!/usr/bin/env bash

#
# Grasp helper
#
# Используется для клонирования bare-репозиториев в каталог
# имя_пользователя/имя_репозитория. 
#
#
#
# Подключаем в .bashrc
# if [ -f /path/to/grasp.sh ]; then
#    source /path/to/grasp.sh
# fi
#
# и выполняем
# source ~/.bashrc
#
# теперь поддерживется команда grasp clone или вызов скрипта напрямую.
#
# (c) Karel Wintersky, 2025
#
 
# Проверка режима выполнения
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    RUN_AS_SCRIPT=true
else
    RUN_AS_SCRIPT=false
fi

# Цвета вывода
RED='\033[1;31m'; GREEN='\033[1;32m'; YELLOW='\033[1;33m'
BLUE='\033[1;34m'; CYAN='\033[1;36m'; NC='\033[0m'

main() {
    case "$1" in
        "clone")
            [[ -z "$2" ]] && { show_error "URL не указан"; show_help; return 1; }
            clone_bare "$2"
            ;;
        "help"|"")
            show_help
            ;;
        *)
            show_error "Неподдерживаемая команда '${YELLOW}$1${RED}'"
            show_help
            return 1
            ;;
    esac
}

clone_bare() {
    local url="$1"
    local user_repo service

    # Определяем сервис и извлекаем user/repo
    case "$url" in
        *github*)
            service="GitHub"
            [[ "$url" =~ github.com[/:]([^/]+/[^/]+?)(\.git)?$ ]] || { show_error "Неверный URL GitHub"; return 1; }
            ;;
        *gitlab*)
            service="GitLab"
            [[ "$url" =~ gitlab.com[/:]([^/]+/[^/]+?)(\.git)?$ ]] || { show_error "Неверный URL GitLab"; return 1; }
            ;;
        *bitbucket*)
            service="Bitbucket"
            [[ "$url" =~ bitbucket.org[/:]([^/]+/[^/]+?)(\.git)?$ ]] || { show_error "Неверный URL Bitbucket"; return 1; }
            ;;
        *)
            show_error "Неподдерживаемый хостинг. Поддерживается: GitHub, GitLab, Bitbucket"
            return 1
            ;;
    esac

    user_repo="${BASH_REMATCH[1]}"
    echo -e "${GREEN}Клонируем ${CYAN}${service}${GREEN} bare-репозиторий в ${YELLOW}$user_repo${NC}..."

    if mkdir -p "$user_repo" && git clone --bare "$url" "$user_repo"; then
        echo -e "${GREEN}Успешно! Репозиторий сохранён в ${YELLOW}$user_repo${NC}"
    else
        show_error "Ошибка при клонировании"
        return 1
    fi
}

show_error() {
    echo -e "${RED}Ошибка: ${1}${NC}" >&2
}

show_help() {
    echo ""
    echo -e "${GREEN}GRASP Helper${NC}" 
    echo ""
    echo -e "  ${YELLOW}grasp clone <URL>${NC}  - Клонировать как bare-репозиторий"
    echo -e "  ${YELLOW}grasp help       ${NC}  - Показать справку"
    echo ""
    echo -e "${BLUE}Поддерживаемые сервисы:${NC}"
    echo ""
    echo -e "  GitHub:    ${CYAN}https://github.com/user/repo${NC}"
    echo -e "             ${CYAN}git@github.com:user/repo.git${NC}"
    echo -e "  GitLab:    ${CYAN}https://gitlab.com/user/repo${NC}"
    echo -e "             ${CYAN}git@gitlab.com:user/repo.git${NC}"
    echo -e "  Bitbucket: ${CYAN}https://bitbucket.org/user/repo${NC}"
    echo -e "             ${CYAN}git@bitbucket.org:user/repo.git${NC}"
    echo ""
}

# Точка входа
if $RUN_AS_SCRIPT; then
    main "$@"
else
    grasp() { main "$@"; }
fi