#!/bin/bash

### Конфигурация ###
SEARCH_EXTENSION="*"  # Можно изменить на "*" для поиска всех файлов
SITES_AVAILABLE="/etc/nginx/sites-available"
SITES_ENABLED="/etc/nginx/sites-enabled"

### Стили вывода ###
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

### Проверка прав ###
check_root() {
    if [ "$(id -u)" -ne 0 ]; then
        echo -e "${RED}Ошибка: требуется root (используйте sudo)${NC}" >&2
        exit 1
    fi
}

### Получить реальный путь из симлинка ###
get_available_path() {
    local enabled_link="$1"
    readlink -f "$enabled_link" | sed "s|^$SITES_AVAILABLE/||"
}

### Поиск конфигов ###
find_configs() {
    local search_pattern="*.$SEARCH_EXTENSION"
    [ "$SEARCH_EXTENSION" = "*" ] && search_pattern="*"
    
    find "$SITES_AVAILABLE" -type f -name "$search_pattern" | while read -r file; do
        echo "${file#$SITES_AVAILABLE/}"
    done
}

### Проверка включен ли сайт ###
is_enabled() {
    local site_path="$1"
    find "$SITES_ENABLED" -type l | while read -r link; do
        if [ "$(get_available_path "$link")" = "$site_path" ]; then
            echo "1"
            return
        fi
    done | grep -q "1"
}

### Получить имя симлинка для сайта ###
get_enabled_name() {
    local site_path="$1"
    find "$SITES_ENABLED" -type l | while read -r link; do
        if [ "$(get_available_path "$link")" = "$site_path" ]; then
            basename "$link"
            return
        fi
    done
}

### Вывод списка ###
list_sites() {
    echo -e "${YELLOW}Доступные сайты (расширение: $SEARCH_EXTENSION):${NC}"
    echo "===================================="
    
    find_configs | while read -r rel_path; do
        if is_enabled "$rel_path"; then
            local enabled_name=$(get_enabled_name "$rel_path")
            local base_name=$(basename "$rel_path")
            if [ "$enabled_name" != "$base_name" ]; then
                echo -e "${GREEN}[enabled]${NC}  $rel_path (как $enabled_name)"
            else
                echo -e "${GREEN}[enabled]${NC}  $rel_path"
            fi
        else
            echo -e "${RED}[disabled]${NC} $rel_path"
        fi
    done
}

### Включение сайта ###
enable_site() {
    local site_path="$1"
    local link_name="${2:-$(basename "$site_path")}"
    local available_path="$SITES_AVAILABLE/$site_path"
    local enabled_path="$SITES_ENABLED/$link_name"

    if [ ! -f "$available_path" ]; then
        echo -e "${RED}Ошибка: файл $available_path не существует${NC}" >&2
        exit 1
    fi

    if is_enabled "$site_path"; then
        local current_name=$(get_enabled_name "$site_path")
        if [ "$current_name" != "$(basename "$site_path")" ]; then
            echo -e "${YELLOW}Предупреждение: сайт уже включен как $current_name${NC}" >&2
        else
            echo -e "${YELLOW}Предупреждение: сайт уже включен${NC}" >&2
        fi
        return
    fi

    if ln -s "$available_path" "$enabled_path"; then
        if [ "$link_name" != "$(basename "$site_path")" ]; then
            echo -e "${GREEN}Включен: $site_path (как $link_name)${NC}"
        else
            echo -e "${GREEN}Включен: $site_path${NC}"
        fi
        
        if ! nginx -t; then
            echo -e "${RED}Ошибка конфигурации!${NC}" >&2
            rm "$enabled_path"
            exit 1
        fi
        
        systemctl reload nginx
        echo "NGINX перезагружен"
    else
        echo -e "${RED}Ошибка создания симлинка${NC}" >&2
        exit 1
    fi
}

### Выключение сайта ###
disable_site() {
    local site_path="$1"
    local enabled_name=$(get_enabled_name "$site_path")
    local enabled_path="$SITES_ENABLED/$enabled_name"

    if [ -z "$enabled_name" ]; then
        echo -e "${YELLOW}Предупреждение: сайт не включен${NC}" >&2
        return
    fi

    if rm "$enabled_path"; then
        if [ "$enabled_name" != "$(basename "$site_path")" ]; then
            echo -e "${GREEN}Выключен: $enabled_name (ссылался на $site_path)${NC}"
        else
            echo -e "${GREEN}Выключен: $site_path${NC}"
        fi
        
        if ! nginx -t; then
            echo -e "${RED}Ошибка конфигурации!${NC}" >&2
            exit 1
        fi
        
        systemctl reload nginx
        echo "NGINX перезагружен"
    else
        echo -e "${RED}Ошибка удаления симлинка${NC}" >&2
        exit 1
    fi
}

### Основной код ###
show_help() {
    echo -e "${YELLOW}Использование:${NC}"
    echo "  nghost list                       - Список сайтов"
    echo "  nghost enable <site> [link_name]  - Включить сайт"
    echo "  nghost disable <site>             - Выключить сайт"
    echo -e "\n${YELLOW}Настройки:${NC}"
    echo "  Измените SEARCH_EXTENSION в скрипте"
    echo "  Доступные значения: conf (по умолчанию) или * (все файлы)"
}

check_root

case "$1" in
    list)
        list_sites
        ;;
    enable)
        if [ -z "$2" ]; then
            echo -e "${RED}Укажите имя сайта${NC}" >&2
            show_help
            exit 1
        fi
        enable_site "$2" "$3"
        ;;
    disable)
        if [ -z "$2" ]; then
            echo -e "${RED}Укажите имя сайта${NC}" >&2
            show_help
            exit 1
        fi
        disable_site "$2"
        ;;
    *)
        show_help
        exit 1
        ;;
esac

