#!/bin/bash

# Скрипт проверки уязвимости CVE-2026-24061 в GNU InetUtils telnetd
# Уязвимые версии: 1.9.3 - 2.7 (май 2015 - январь 2026)
# by Claude Code 4.5

RED='\033[0;31m'
YELLOW='\033[1;33m'
GREEN='\033[0;32m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

echo -e "${BLUE}========================================${NC}"
echo -e "${BLUE}Проверка уязвимости CVE-2026-24061${NC}"
echo -e "${BLUE}GNU InetUtils telnetd${NC}"
echo -e "${BLUE}========================================${NC}\n"

VULNERABLE=0

# 1. Проверка наличия telnetd
echo -e "${BLUE}[1] Поиск telnetd в системе...${NC}"

TELNETD_PATHS=$(which telnetd 2>/dev/null)
if [ -z "$TELNETD_PATHS" ]; then
    TELNETD_PATHS=$(find /usr/sbin /sbin /usr/local/sbin -name "telnetd" 2>/dev/null)
fi

if [ -z "$TELNETD_PATHS" ]; then
    echo -e "${GREEN}✓ telnetd не найден в системе${NC}\n"
else
    echo -e "${YELLOW}⚠ Найден telnetd:${NC}"
    echo "$TELNETD_PATHS" | while read path; do
        echo "  - $path"
    done
    echo ""
fi

# 2. Проверка версии GNU InetUtils
echo -e "${BLUE}[2] Проверка версии GNU InetUtils...${NC}"

for telnetd_bin in $TELNETD_PATHS; do
    if [ -f "$telnetd_bin" ]; then
        # Попытка определить версию
        VERSION_OUTPUT=$($telnetd_bin --version 2>&1 | head -n 1)
        
        echo -e "${YELLOW}Файл: $telnetd_bin${NC}"
        echo "Вывод --version: $VERSION_OUTPUT"
        
        # Проверка, что это GNU InetUtils
        if echo "$VERSION_OUTPUT" | grep -qi "inetutils"; then
            echo -e "${YELLOW}✓ Обнаружен GNU InetUtils telnetd${NC}"
            
            # Извлечение версии
            VERSION=$(echo "$VERSION_OUTPUT" | grep -oP '\d+\.\d+(\.\d+)?' | head -n 1)
            
            if [ -n "$VERSION" ]; then
                echo "Версия: $VERSION"
                
                # Проверка диапазона уязвимых версий (1.9.3 - 2.7)
                MAJOR=$(echo $VERSION | cut -d. -f1)
                MINOR=$(echo $VERSION | cut -d. -f2)
                PATCH=$(echo $VERSION | cut -d. -f3)
                
                # Уязвимы версии >= 1.9.3 и <= 2.7
                if [ "$MAJOR" -eq 1 ] && [ "$MINOR" -eq 9 ] && [ "${PATCH:-0}" -ge 3 ]; then
                    echo -e "${RED}✗ УЯЗВИМАЯ ВЕРСИЯ ОБНАРУЖЕНА!${NC}"
                    VULNERABLE=1
                elif [ "$MAJOR" -eq 1 ] && [ "$MINOR" -gt 9 ]; then
                    echo -e "${RED}✗ УЯЗВИМАЯ ВЕРСИЯ ОБНАРУЖЕНА!${NC}"
                    VULNERABLE=1
                elif [ "$MAJOR" -eq 2 ] && [ "$MINOR" -le 7 ]; then
                    echo -e "${RED}✗ УЯЗВИМАЯ ВЕРСИЯ ОБНАРУЖЕНА!${NC}"
                    VULNERABLE=1
                else
                    echo -e "${GREEN}✓ Версия не входит в уязвимый диапазон${NC}"
                fi
            else
                echo -e "${YELLOW}⚠ Не удалось определить версию${NC}"
            fi
        else
            echo -e "${GREEN}✓ Это не GNU InetUtils telnetd${NC}"
        fi
        echo ""
    fi
done

# 3. Проверка активности telnet-сервиса
echo -e "${BLUE}[3] Проверка активных telnet-сервисов...${NC}"

# Проверка через systemctl
if command -v systemctl &> /dev/null; then
    TELNET_SERVICE=$(systemctl list-units --all | grep -i telnet | grep -v grep)
    if [ -n "$TELNET_SERVICE" ]; then
        echo -e "${YELLOW}⚠ Обнаружены telnet-сервисы:${NC}"
        echo "$TELNET_SERVICE"
        
        # Проверка активности
        if systemctl is-active --quiet telnet.socket || systemctl is-active --quiet telnetd; then
            echo -e "${RED}✗ telnet-сервис АКТИВЕН!${NC}"
            VULNERABLE=1
        fi
    else
        echo -e "${GREEN}✓ telnet-сервисы не обнаружены в systemd${NC}"
    fi
fi

# Проверка прослушиваемых портов
echo ""
TELNET_PORT=$(netstat -tuln 2>/dev/null | grep ":23 " || ss -tuln 2>/dev/null | grep ":23 ")
if [ -n "$TELNET_PORT" ]; then
    echo -e "${RED}✗ Порт 23 (telnet) прослушивается!${NC}"
    echo "$TELNET_PORT"
    VULNERABLE=1
else
    echo -e "${GREEN}✓ Порт 23 не прослушивается${NC}"
fi

# 4. Проверка inetd/xinetd
echo ""
echo -e "${BLUE}[4] Проверка inetd/xinetd конфигурации...${NC}"

if [ -f /etc/inetd.conf ]; then
    TELNET_INETD=$(grep -v "^#" /etc/inetd.conf | grep telnet)
    if [ -n "$TELNET_INETD" ]; then
        echo -e "${YELLOW}⚠ telnet включен в /etc/inetd.conf:${NC}"
        echo "$TELNET_INETD"
    else
        echo -e "${GREEN}✓ telnet не найден в /etc/inetd.conf${NC}"
    fi
fi

if [ -d /etc/xinetd.d/ ]; then
    TELNET_XINETD=$(grep -r "disable.*=.*no" /etc/xinetd.d/telnet 2>/dev/null)
    if [ -n "$TELNET_XINETD" ]; then
        echo -e "${YELLOW}⚠ telnet может быть включен в xinetd${NC}"
    else
        echo -e "${GREEN}✓ telnet не активен в xinetd${NC}"
    fi
fi

# 5. Итоговый отчет
echo ""
echo -e "${BLUE}========================================${NC}"
echo -e "${BLUE}ИТОГОВЫЙ ОТЧЕТ${NC}"
echo -e "${BLUE}========================================${NC}"

if [ $VULNERABLE -eq 1 ]; then
    echo -e "${RED}✗ СИСТЕМА УЯЗВИМА!${NC}"
    echo ""
    echo -e "${YELLOW}Рекомендации:${NC}"
    echo "1. Немедленно обновите GNU InetUtils до версии > 2.7"
    echo "2. Или полностью отключите telnetd:"
    echo "   - systemctl stop telnet.socket"
    echo "   - systemctl disable telnet.socket"
    echo "   - apt remove inetutils-telnetd (Debian/Ubuntu)"
    echo "   - yum remove telnet-server (RHEL/CentOS)"
    echo "3. Используйте SSH вместо telnet"
    echo "4. Проверьте логи на предмет попыток эксплуатации"
else
    echo -e "${GREEN}✓ Система не уязвима к CVE-2026-24061${NC}"
    echo ""
    echo "Либо уязвимое ПО не установлено,"
    echo "либо установлена безопасная версия."
fi

echo -e "${BLUE}========================================${NC}"