#!/usr/bin/env python3
import os
import datetime
from pathlib import Path

def get_user_input():
    """Получить входные данные от пользователя"""
    print("=== Генератор файлов для каталога debian ===\n")
    
    package_name = input("Введите имя пакета: ").strip()
    description = input("Введите описание пакета: ").strip()
    
    default_version = "0.0.1"
    package_version = input(f"Введите версию пакета [{default_version}]: ").strip()
    if not package_version:
        package_version = default_version
    
    default_author = "Karel Wintersky"
    author_name = input(f"Введите имя автора [{default_author}]: ").strip()
    if not author_name:
        author_name = default_author
    
    default_email = "karel.wintersky@yandex.ru"
    author_email = input(f"Введите email автора [{default_email}]: ").strip()
    if not author_email:
        author_email = default_email
    
    return {
        'package_name': package_name,
        'description': description,
        'package_version': package_version,
        'author_name': author_name,
        'author_email': author_email,
        'current_date': datetime.datetime.now().strftime("%a, %d %b %Y %H:%M:%S %z")
    }

def generate_control_file(params):
    """Сгенерировать файл control"""
    content = f"""Package: {params['package_name']}
Version: {params['package_version']}
Section: utils
Priority: optional
Architecture: all
Depends: php8.0, apache2 | nginx | lighttpd
Maintainer: {params['author_name']} <{params['author_email']}>
Description: {params['description']}
"""
    return content

def generate_rules_file(params):
    """Сгенерировать файл rules"""
    content = f"""#!/usr/bin/make -f

# output every command that modifies files on the build system.

# export DH_VERBOSE = 1

%:
\tdh $@


override_dh_auto_install:
\t$(MAKE) install DESTDIR=$(CURDIR)/debian/{params['package_name']}

override_dh_auto_build:


"""
    return content

def generate_postinst_file(params):
    """Сгенерировать файл postinst"""
    content = f"""#! /bin/sh

set -e

WWW_ROOT=/var/www/{params['package_name']}/

case "$1" in
    configure)

      if [ -d $WWW_ROOT ]; then
        chown -R www-data:www-data $WWW_ROOT/*
        exit 0
      fi

      echo "postinst can't find path: $WWW_ROOT!" >&2
      exit 1
    ;;

    abort-upgrade|abort-remove|abort-deconfigure)

    ;;

    *)
        echo "postinst called with unknown argument \\`$1'" >&2
        exit 1
    ;;
esac

#DEBHELPER#

exit 0
"""
    return content

def generate_copyright_file(params):
    """Сгенерировать файл copyright"""
    content = f"""Format: https://www.debian.org/doc/packaging-manuals/copyright-format/1.0/
Upstream-Name: {params['package_name']}
Source: -

This package was debianized by {params['author_name']} <{params['author_email']}> on
{params['current_date']}

Upstream Author(s): {params['author_name']} <{params['author_email']}>

Copyright:

<Вставьте сюда информацию о копирайте>
"""
    return content

def generate_source_format():
    """Сгенерировать файл source/format"""
    return "3.0 (native)"

def generate_compat_file():
    """Сгенерировать файл compat"""
    return "10"

def generate_dirs_file():
    """Сгенерировать пустой файл dirs"""
    return ""

def generate_cron_file(params):
    """Сгенерировать файл <packagename>.cron.d"""
    content = f"""#
# Regular cron jobs for the {params['package_name']} package
#
"""
    return content

def write_file(filename, content):
    """Записать содержимое в файл"""
    Path(filename).parent.mkdir(parents=True, exist_ok=True)
    
    with open(filename, 'w', encoding='utf-8') as f:
        f.write(content)
    print(f"Создан файл: {filename}")

def main():
    # Получить входные данные
    params = get_user_input()
    
    # Создать директорию debian если её нет
    debian_dir = "debian"
    os.makedirs(debian_dir, exist_ok=True)
    
    # Сгенерировать и записать файлы
    files_to_generate = [
        (f"{debian_dir}/control", generate_control_file(params)),
        (f"{debian_dir}/rules", generate_rules_file(params)),
        (f"{debian_dir}/postinst", generate_postinst_file(params)),
        (f"{debian_dir}/copyright", generate_copyright_file(params)),
        (f"{debian_dir}/source/format", generate_source_format()),
        (f"{debian_dir}/compat", generate_compat_file()),
        (f"{debian_dir}/dirs", generate_dirs_file()),
        (f"{debian_dir}/{params['package_name']}.cron.d", generate_cron_file(params)),
    ]
    
    print(f"\n=== Генерация файлов в директории '{debian_dir}/' ===")
    
    for filename, content in files_to_generate:
        write_file(filename, content)
    
    # Сделать rules исполняемым, postinst - НЕ исполняемым
    os.chmod(f"{debian_dir}/rules", 0o755)
    # postinst оставляем без атрибута X (обычный файл)
    
    print(f"\n=== Готово! Все файлы созданы в директории '{debian_dir}/' ===")

if __name__ == "__main__":
    main()

