#!/usr/bin/env python3
import sys
import os
import argparse
from pathlib import Path

def fix_mojibake_utf8(input_path, output_path=None):
    """
    Исправляет текст, который был сохранён как UTF-8, но прочитан как ISO-8859-1
    """
    try:
        # Читаем файл как бинарный
        with open(input_path, 'rb') as f:
            raw_bytes = f.read()
        
        # Исправляем mojibake
        fixed_text = raw_bytes.decode('utf-8').encode('latin-1').decode('utf-8')
        
        return fixed_text
        
    except UnicodeDecodeError:
        # Если не получается исправить как mojibake, пробуем прочитать как обычный UTF-8
        try:
            with open(input_path, 'r', encoding='utf-8') as f:
                return f.read()
        except:
            with open(input_path, 'r', encoding='latin-1') as f:
                return f.read()
    except Exception as e:
        print(f"✗ Ошибка при обработке {input_path}: {e}")
        return None

def process_file(input_file, output_file):
    """Обрабатывает один файл"""
    fixed_text = fix_mojibake_utf8(input_file)
    if fixed_text is None:
        return False
    
    # Записываем исправленный текст
    with open(output_file, 'w', encoding='utf-8') as f:
        f.write(fixed_text)
    
    return True

def process_directory(input_dir, prefix="_"):
    """Обрабатывает все HTML файлы в каталоге"""
    input_dir = Path(input_dir)
    
    if not input_dir.exists():
        print(f"✗ Каталог не найден: {input_dir}")
        return False
    
    if not input_dir.is_dir():
        print(f"✗ Это не каталог: {input_dir}")
        return False
    
    # Ищем все HTML файлы
    html_files = list(input_dir.glob("*.html")) + list(input_dir.glob("*.htm"))
    
    if not html_files:
        print(f"✗ В каталоге {input_dir} не найдено HTML файлов")
        return False
    
    print(f"Найдено {len(html_files)} HTML файлов для обработки")
    print("-" * 50)
    
    processed = 0
    for input_file in html_files:
        # Создаем имя выходного файла с префиксом
        output_file = input_file.with_name(f"{prefix}{input_file.name}")
        
        print(f"Обработка: {input_file.name} → {output_file.name}")
        
        if process_file(input_file, output_file):
            processed += 1
        else:
            print(f"  ✗ Ошибка при обработке {input_file.name}")
    
    print("-" * 50)
    print(f"✓ Обработано файлов: {processed}/{len(html_files)}")
    return True

def main():
    parser = argparse.ArgumentParser(
        description='Исправляет mojibake (UTF-8, прочитанный как Latin-1) в файлах'
    )
    parser.add_argument('input', help='Входной файл или каталог')
    parser.add_argument('output', nargs='?', help='Выходной файл (только для обработки одного файла)')
    parser.add_argument('--prefix', default='_', help='Префикс для обработанных файлов (по умолчанию: _)')
    
    args = parser.parse_args()
    
    input_path = Path(args.input)
    
    # Проверяем, является ли вход каталогом
    if input_path.is_dir():
        print(f"📁 Обработка каталога: {input_path}")
        process_directory(input_path, args.prefix)
    else:
        # Обработка одного файла
        if not input_path.exists():
            print(f"✗ Файл не найден: {input_path}")
            sys.exit(1)
        
        if not args.output:
            print("✗ Для обработки одного файла нужно указать выходной файл")
            print("Использование: python script.py input.html output.html")
            sys.exit(1)
        
        print(f"📄 Обработка файла: {input_path}")
        if process_file(input_path, args.output):
            print(f"✓ Файл сохранен: {args.output}")
            
            # Показываем превью
            with open(args.output, 'r', encoding='utf-8') as f:
                content = f.read(500)
                print("\nПревью (первые 500 символов):")
                print("-" * 50)
                print(content)
                print("-" * 50)
        else:
            print(f"✗ Не удалось обработать файл")

if __name__ == "__main__":
    main()